 /**
 *
 * \file        HPRFGW_RFRecMsg.c
 *
 * \brief       This file contains the RF receive message processing routines.
 *
 * \detail
 *
 * \author      Hazrat Shah
 *
 * \date        10/11/2006
 * \warning
 * \note
 * \review
 */
#include "hprfgw_config.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfApp.h"
#include "hprfgw_rfDesc.h"
#include "hprfgw_rfComLog.h"
#include "hprfgw_rfResultCode.h"
#include "hprfgw_rfRetry.h"
#include "hprfgw_rfHeartbeat.h"
#include "hprfgw_rfRec.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfPkt.h"
#include "hprfgw_rfRecMsg.h"
#include "hprfgw_rfSlaveIntToRfApp.h"
#include "hprfgw_rfSharedDefs.h"
#include "hprfgw_rfSlaveIntToHost.h"

#ifdef  RF_NETWORK_MASTER_DEVICE
  #include "network.h"
  #include "image_server.h"
  #include "coordinating_task.h"
  #include "ip_if.h"
  #include "hprfgw_console.h"
  #include "hprfgw_cresnet.h"
  #include "image_transmission_task.h"
  #include "hprfgw_hid.h"
#endif

#ifdef  RF_NETWORK_MASTER_DEVICE
extern BOOL IsSlaveDevicePassToActive(void);
#endif

////////////////////////////////////////////////////////////////////////////////
//  DEFINITIONS
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//  LOCAL DATA
////////////////////////////////////////////////////////////////////////////////
UINT8   OptionFlags;

////////////////////////////////////////////////////////////////////////////////
//  external functions
////////////////////////////////////////////////////////////////////////////////
extern  INT8 *ConvertUIDtoString (UINT8 *puid, INT8 *pstr, UINT8 addrmode);
extern  INT8 *ConvertLinkStatetoString (UINT8 linkstate, INT8 *pbuf);

////////////////////////////////////////////////////////////////////////////////
//  LOCAL FUNCTIONS
///////////////////////////////////////////////////////////////////////////////
void RFRecMsg_SaveOptionFlags (UINT8 options);

/**
 *
 * \author      Hazrat Shah
 *
 * \brief
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       None
 *
 * \return
 *
 * \retval
 */
UINT16 ConvUINTToBigIndian    (UINT8 *data) {
#if defined (HPRF_BIG_ENDIAN)
	return ((*(data+1)*256)+*data);
#else
	return (*(HPRF_UNALIGNED UINT16*)data);
#endif
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       Process received options flags
 *
 * \detail
 *
 * \date        05/31/2007
 *
 * \param       UINT8 options
 *
 * \return      None
 *
 * \retval
 */
void    RFRecMsg_ProcessOptions (UINT8 options) {

#ifdef  RF_NETWORK_MASTER_DEVICE
    //  Master device (gateway)

#else
    RFRecMsg_SaveOptionFlags (options);
    //  control status change is only accepted if acquire is not running
	if ((!RFAPP_AcquireStatus()) && RFReceive_IsGatewayConnected()) {
        //  save connection type Cresnet/Ethernet
        if (options & RF_HEADER_EXPANSION_CS_CONNECT_STATE) {
            if (!RFReceive_IsControlSystemConnected())  {
                RFReceive_SetControlSystemConnectedStatus(TRUE, FALSE);
                RFReceive_SetControlSystemConnectedNetTypeCresnet ((options & RF_HEADER_EXPANSION_CRESNET_NET)?TRUE:FALSE);
                RFReceive_SetControlSystemConnectedNetTypeEthernet((options & RF_HEADER_EXPANSION_ETHERNET_NET)?TRUE:FALSE);
            }
        }   else    {
            if (RFReceive_IsControlSystemConnected())   {
                RFReceive_SetControlSystemConnectedStatus(FALSE, FALSE);
                RFReceive_SetControlSystemConnectedNetTypeCresnet (FALSE);
                RFReceive_SetControlSystemConnectedNetTypeEthernet(FALSE);
            }
        }
    }
#endif
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       Swap high/low bytes
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT16 value
 *
 * \return      UINT16
 *
 * \retval
 */
UINT16 ConvUINTToLittleIndian (UINT16 value)    {
#if defined (HPRF_BIG_ENDIAN)
	return (((value%256)<<8)+(value/256));
#else
	return (value);
#endif
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function submits the packet to the rf application
 *
 * \detail
 * \date        07/09/2007
 *
 * \param       UINT8 index, UINT8* pBuf, UINT16 len
 *
 * \return      none
 *
 * \retval
 */
void    RFRecMsg_SubmitPacketToRFApplication (UINT8 index, UINT8* pBuf, UINT16 len) {
    TYPE_RFAPP_DATA_BUF rfapp;

    //  forward data field reply packet to the rf application task
    rfapp.inthdr.index = index;
    rfapp.inthdr.rfid  = g_pRFDeviceDescriptiveTable[index].IDInfo.rfid;
    memcpy(&rfapp.pkthdr, pBuf, len);
    RFAPP_PostToPipe((UINT8*)&rfapp, (sizeof(rfapp)+len)-(sizeof(rfapp.data)+sizeof(TYPE_INTERNAL_RF_PACKET_HEADER)));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       process local cresnet packets
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT8 index, UINT8 *pBuf, UINT16 Len
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 */
UINT16   RFRecMsg_ProcessLocalCresnetPackets (UINT8 index, UINT8 *pBuf, UINT16 Len) {
    const TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;
    UINT16  res     = SUCCESS;
#ifdef  RF_NETWORK_MASTER_DEVICE
    UINT8   type    = *(pBuf+2);
    UINT8   subtype = *(pBuf+3);
    UINT8   mode    = *(pBuf+4);
    UINT8   action  = *(pBuf+5);

    INT8    buf[200];
    UINT16  i;
    UINT8   rfid;
    UINT8   indexarray[MAX_DESCRIPTIVE_TABLE_ENTRIES];

    INT32(*pcbf)(UINT32 par, UINT32 status) = NULL;
#endif

#if 0
    //  Cresnet packet dump
    ConsoleHexDumpPacket((INT8*)pBuf, Len, "LocalCresnet RX via RF:", 1);
#endif

#ifdef  RF_NETWORK_MASTER_DEVICE

    //  All Local Cresnet packets for master packets are proccessed here
    switch (type)   {
        case    RFS_MESH_NET:
            if          (subtype == RFS_MNET_SUBTYPE_INFO)  {
                if ((mode == RFS_MNET_MODE_IDENTITY_STRING) && (action == RFS_MNET_ACTION_ID_STRING_REPLY)) {
                    //  update the device tsid, rfid, idstring in the descriptive table
                    //  [ID] [L] 28 00 00 02 [TSID HI] [TSID] [TSID] [TSID LO] [MNET ID] 00 00 <standard id text>
                    memcpy ((INT8*)&g_pRFDeviceDescriptiveTable[index].IDInfo.tsid, pBuf+6, 4);
                    rfid = g_pRFDeviceDescriptiveTable[index].IDInfo.rfid;
                    g_pRFDeviceDescriptiveTable[index].IDInfo.rfid  = *(pBuf+10);
                    RFDesc_UpdateDeviceIDStr (index, pBuf+13, *(pBuf+1)-11);
                    //  Update the device list by RFID
                    RFDesc_BuildActiveRFDeviceTableIndexByRFID();
                    if (g_pRFDeviceDescriptiveTable[index].IDInfo.rfid != rfid) {
                        SetCresnetAnalogSubscriptionData (ANALOG_JOIN_FIRST_RFDEV_LINK_STATE+g_pRFDeviceDescriptiveTable[index].IDInfo.rfid, g_pRFDeviceDescriptiveTable[index].linkstate, FALSE);
                        if (RFDesc_GetActiveRFDeviceTableIndexByRFID(rfid, indexarray) == 0)    {
                            //  report the previous device is FAIL
                            SetCresnetAnalogSubscriptionData (ANALOG_JOIN_FIRST_RFDEV_LINK_QUALITY_AT_HPRFGW+rfid, 0, FALSE);
                            SetCresnetAnalogSubscriptionData (ANALOG_JOIN_FIRST_RFDEV_LINK_QUALITY_AT_DEV+rfid,    0, FALSE);
                            SetCresnetAnalogSubscriptionData (ANALOG_JOIN_FIRST_RFDEV_LINK_STATE+rfid, RF_DEVICE_LINK_STATE_FAIL, FALSE);
                        }
                    }
                    //  Forward the string to the console (this string is received by the slave to the master as a result of device id string request packet)
                    indexarray[0] = index;
                    ShowDevicesRFNetIDString (indexarray, 1, FALSE, TRUE);
                }
            }   else if (subtype == RFS_MNET_SUBTYPE_TSID_EXTENSION)    {
                if (mode == RFS_MNET_MODE_LIGHT_AND_POLL_BUTTON_PRESSED)    {
                    //  take device out of light by ppn mode
                    buf[0]  = g_pRFDeviceDescriptiveTable[index].IDInfo.rfid;
                    buf[1]  = 4;
                    buf[2]  = RFS_MESH_NET;
                    buf[3]  = RFS_MNET_SUBTYPE_TSID_EXTENSION;
                    buf[4]  = RFS_MNET_MODE_TSID;
                    buf[5]  = RFS_MNET_ACTION_CLR_DEVICE_LIGHT_AND_POLL_MODE_WITH_CFM;
                    RFTransmit_BuildLocalCresnetPacketAndPostToPipeByIndex (index, (UINT8*) buf, buf[1]+2, pcbf, 0, 0);
                    //  show light and poll response in the console
                    ShowDeviceLightByTSIDString (&index, 1);
                }
            }
            break;
    case CRESNET_SERIAL:
      {
        UINT16 serial_join;

        serial_join = *(UINT16 *)(&pBuf[3]) + 1;
        // check to see if this is the Image Transfer Reserved Serial Join
        if (serial_join == SYSTEM_DYNAMIC_GRAPHICS_DATA_PIPE)
          {
            if (pBuf[6] == 0x00)  // if this is a begin retrieval request
              {
                request_image_registration(g_pRFDeviceDescriptiveTable[index].IDInfo.rfid,  // panel_ID
                                            *(unsigned long *)&pBuf[7],                     // panels_image_identifier
                                            *(unsigned short *)&pBuf[11],                   // width
                                            *(unsigned short *)&pBuf[13],                   // height
                                            (struct image_signature *)&pBuf[15],            // signature
                                            *(unsigned long *)&pBuf[23],                    // update_period
                                            *(unsigned long *)&pBuf[27],                    // timeout_period
                                            (char *)&pBuf[31]);                             // URL

                if ( DebugTraceIsImageServerTraceEnabled() )
                  {
                    RFIntHostApp.pF_ConsolePrintf("REQUEST:\r\n");
                    RFIntHostApp.pF_ConsolePrintf("Token %d\r\n", *(unsigned long *)&pBuf[7]);
                    RFIntHostApp.pF_ConsolePrintf("Width %d\r\n", *(unsigned short *)&pBuf[11]);
                    RFIntHostApp.pF_ConsolePrintf("Height %d\r\n", *(unsigned short *)&pBuf[13]);
                    RFIntHostApp.pF_ConsolePrintf("URL %s\r\n", &pBuf[31]);
                  }
              }
            if (pBuf[6] == 0x01)  // if this is a end retrieval request
              {
                request_image_un_registration(g_pRFDeviceDescriptiveTable[index].IDInfo.rfid,  // panel_ID
                                                *(unsigned long *)&pBuf[7]);                    // panels_image_identifier
              }
            if (pBuf[6] == 0x02)  // if image processing completed message
              {
                report_image_processed(g_pRFDeviceDescriptiveTable[index].IDInfo.rfid,  // panel_ID
                                                *(unsigned long *)&pBuf[7]);                    // panels_image_identifier
              }
          }
          /* HID message from TPS-6X */
          else if (serial_join == SYSTEM_HID_HPRFGW_MSG_SERIAL_JOIN)
          {
              HID_PostMessageToPipe(pBuf, Len);
          }
        break;
      }
    default:
            //  unexpected Cresnet packet
            for (i=0; i<Len && i<32; i++)   {   sprintf (&buf[i], "%02x", *(pBuf+i));   }
            RFIntHostApp.pF_ErrorPrintf ("Unexpected Cresnet packet [%s]\r\n", buf);
            res = FAILURE;
            break;
    }
#else
    //  All Local Cresnet packets for slave devices are submitted to the slave host application
    res = (pHF->pF_ProcessLocalCresnetPacket)(pBuf, Len);
#endif

    return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       process device communication error packet
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT8 index, TYPE_RF_COMMUNICATION_ERROR_PACKET_DATA *pCE
 *
 * \return      void
 *
 * \retval
 */
void    RFRecMsg_ProcessCommunicationErrorPacket (UINT8 index, TYPE_RF_COMMUNICATION_ERROR_PACKET_DATA *pCE)
{
    //BOOL    isTraceEnabled;
    TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);

    //RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);
    //if (isTraceEnabled)   {
    //  RFIntHostApp.pF_ConsolePrintf ("Communication error packet received: descriptive entry [%02x] RFID[%02x] flags[%04x] expectedseqno[%02x] recseqno[%02x] totalseg[%02x] expectedseg[%02x] recsegno[%02x]\r\n",
    //                                  index, pdst->IDInfo.rfid, pCE->flags, pCE->expectedsequenceno, pCE->receivedsequenceno, pCE->totalsegments,
    //                                  pCE->expectedsegment, pCE->receivedsegment);
    //}

    //HPRF_NTOHS(pCE->flags);
	RFIntHostApp.pF_ErrorPrintf ("Communication error packet received: descriptive entry [%02x] RFID[%02x] flags[%04x] expectedseqno[%02x] recseqno[%02x] totalseg[%02x] expectedseg[%02x] recsegno[%02x]\r\n",
                                        index, pdst->IDInfo.rfid, pCE->flags, pCE->expectedsequenceno, pCE->receivedsequenceno, pCE->totalsegments,
                                        pCE->expectedsegment, pCE->receivedsegment);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       process control system connection packet
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT8 index, TYPE_RF_CS_STATUS_PACKET_DATA *pCS
 *
 * \return      void
 *
 * \retval
 */
void    RFRecMsg_ProcessControlSysConnectPacket (UINT8 index, TYPE_RF_CS_STATUS_PACKET_DATA *pCS)   {
    BOOL    isTraceEnabled;

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);
    if (isTraceEnabled) {
        RFIntHostApp.pF_ConsolePrintf ("Control system status packet: status[%02x]\r\n", RFReceive_IsControlSystemConnected());
    }
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       process device heart beat packet
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT8 index, TYPE_RF_HEART_BEAT_PACKET_DATA *pHB
 *
 * \return      void
 *
 * \retval
 */
void    RFRecMsg_ProcessHeartBeatPacket (UINT8 index, TYPE_RF_HEART_BEAT_PACKET_DATA *pHB)  {
    BOOL    isTraceEnabled;
    TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

    if (isTraceEnabled) {
        //  The device heart beat timer should have been updated on packet reception, no further handling needed
        RFIntHostApp.pF_ConsolePrintf ("Heart Beat Packet received: index[%02x] rfid[%02x]\r\n", index, pdst->IDInfo.rfid);
    }
    //  update remore device link quality
    RFDesc_UpdateLinkQualityOfRemoteDevice (index, pHB->linkquality);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       process echo reply packet
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT8 index, UINT8 *pBuf, UINT16 Len
 *
 * \return      void
 *
 * \retval
 */
void    RFRecMsg_ProcessEchoReplyPacket (UINT8 index, UINT8 *pBuf, UINT16 Len)
{
    //TYPE_RFAPP_DATA_BUF rfapp;
    TYPE_RF_ECHO_REPLY_PACKET_DATA *pER = (TYPE_RF_ECHO_REPLY_PACKET_DATA*) (pBuf+sizeof(TYPE_INTERNAL_RF_PACKET_HEADER));

    //  Store remote device link quality
    RFDesc_UpdateLinkQualityOfRemoteDevice (index, pER->linkquality);

    //  Forward the echo reply packet to the rf application task for further analises
    RFRecMsg_SubmitPacketToRFApplication (index, pBuf, Len);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       process echo request packet
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT8 index, TYPE_RF_ECHO_REQUEST_PACKET_DATA *preq, UINT8 Len
 *
 * \return      void
 *
 * \retval
 */
void    RFRecMsg_ProcessEchoRequestPacket (UINT8 index, TYPE_RF_ECHO_REQUEST_PACKET_DATA *preq, UINT16 Len) {
    //  Store remote device link quality
    RFDesc_UpdateLinkQualityOfRemoteDevice (index, preq->linkquality);
    //  post Echo reply
    RFTransmit_BuildEchoReplyPacketAndPostToPipeByIndex (index, (UINT8*) &preq->data, Len-sizeof(preq->linkquality));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function process the gateway restart message, this message is sent by the
 *              gateway when it restarts
 *
 * \detail
 * \date        07/09/2007
 *
 * \param       UINT8 index, TYPE_RF_GATEWAY_RESTART_PACKET_DATA *pdata
 *
 * \return      none
 *
 * \retval
 */
void    RFRecMsg_ProcessGatewayReStartPacket (UINT8 index, TYPE_RF_GATEWAY_RESTART_PACKET_DATA *pdata)  {
    g_pRFDeviceDescriptiveTable[index].COMLog.rxgwrestarts += 1;
	if (!RFAPP_AcquireStatus()) {
        RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_FAIL);
        RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_CONNECTING);
		RFAPP_ConnectionFailedwithGateway();
    }
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       process buffer overflow packet
 *
 * \detail
 * \date        07/13/2007
 *
 * \param       index, TYPE_RF_BUFFER_OVERFLOW_PACKET_DATA *povdata
 *
 * \return      void
 *
 * \retval
 */
void RFRecMsg_ProcessBufferOverFlowPacket (UINT8 index, TYPE_RF_BUFFER_OVERFLOW_PACKET_DATA *povdata)   {
    BOOL    isTraceEnabled;

    g_pRFDeviceDescriptiveTable[index].COMLog.rxbufoverflowcount += 1;

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

    if (isTraceEnabled) {
        RFIntHostApp.pF_ConsolePrintf ("Buffer overflow packet received: index[%02x]\r\n", index);
    }
    RFIntHostApp.pF_BufferOverFlows(povdata->transmitpipeoverflow, povdata->receivepipeoverflow);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       process network services packet
 *
 * \detail
 * \date        07/13/2007
 *
 * \param       UINT8 index, TYPE_RF_NETWORK_SERIVCES_PACKET_DATA *pns, UINT16 len
 *
 * \return      INT32
 *
 * \retval
 */
INT32   RFRecMsg_ProcessNetworkServicesPacket (UINT8 index, UINT8 *pBuf, UINT16 len)    {
    UINT32  res = SUCCESS;
    UINT32  flags;
    TYPE_RF_NETWORK_SERIVCES_PACKET_DATA *pns = (TYPE_RF_NETWORK_SERIVCES_PACKET_DATA*)(pBuf+sizeof(TYPE_INTERNAL_RF_PACKET_HEADER));
#ifdef	RF_NETWORK_MASTER_DEVICE
	UINT8   uid[UID_MAX_SIZE];
    UINT8   addrmode;
    UINT16  panid;
	BOOL    valid;
	TYPE_RFAPP_DATA_BUF	rfapp;
	INT16 msgLen;
#endif

#ifndef	RF_NETWORK_MASTER_DEVICE
	BOOL    isTraceEnabled;
    UINT8 indexInRoamingTable;

	RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);
#endif

    switch (pns->msgtype)   {
	case    RF_PACKET_DISCOVER_NETDEV_REQUEST:
        #ifdef  RF_NETWORK_MASTER_DEVICE
            if ((pns->pkt.adreq.flags & RF_DISCOVER_GATEWAYS) && RFAPP_IsNetworkServiceEnabled(RF_PACKET_DISCOVER_NETDEV_REPLY))    {
				flags = RFDISCOVER_MASTER_MODE_FLAG|((RFAPP_AcquireStatus())?RFDISCOVER_IN_ACQUIRE_FLAG:0);
                RFTransmit_BuildRFNetDiscoverReplyPacketAndPostToPipeByIndex (index, flags);
            }
        #else
            if ((pns->pkt.adreq.flags & RF_DISCOVER_SLAVES) && RFAPP_IsNetworkServiceEnabled(RF_PACKET_DISCOVER_NETDEV_REPLY))  {
				flags = (RFAPP_AcquireStatus())?RFDISCOVER_IN_ACQUIRE_FLAG:0;
                RFTransmit_BuildRFNetDiscoverReplyPacketAndPostToPipeByIndex (index, flags);
            }
        #endif
            break;

        case    RF_PACKET_DISCOVER_NETDEV_REPLY:
            RFRecMsg_SubmitPacketToRFApplication (index, pBuf, len);
            break;
		case   RF_PACKET_DISCOVER_NETDEV_BEACON_REQ:
		#ifdef  RF_NETWORK_MASTER_DEVICE
			if ( g_roamingEnabled && (pns->pkt.beacon.flags & RF_DISCOVER_GATEWAYS) )    {
				flags = RFDISCOVER_MASTER_MODE_FLAG|((RFAPP_AcquireStatus())?RFDISCOVER_IN_ACQUIRE_FLAG:0);
                RFTransmit_BuildRFNetBeaconReplyPacketAndPostToPipeByIndex (index, flags);
				RFDesc_DeleteBroadcastDeviceUID(g_pRFDeviceDescriptiveTable[index].IDInfo.uid,g_pRFDeviceDescriptiveTable[index].IDInfo.panid);
            }
		#endif
			break;
        case     RF_PACKET_REMOTE_SETUP:
#ifdef  RF_NETWORK_REMOTE_SETUP
    #ifdef  RF_NETWORK_MASTER_DEVICE

    #else
                RFIntHostApp.pF_GetLocalDeviceUID_PANID (&valid, uid, &panid, (UINT8 *)SLAVE_DEVICE_DEFAULT_UID, SLAVE_DEVICE_DEFAULT_PANID);
                // process cmd if device uid and my uid matches
                if (!memcmp(pns->pkt.rmts.deviceuid, uid, UID_MAX_SIZE) && (pns->pkt.rmts.devicepanid == panid) && (pns->pkt.rmts.deviceaddrmode == gAddrModeLong_c)) {
                    if (pns->pkt.rmts.flags & RF_REMOTE_NET_CONNECT)    {
                        // accept new connecion parameters
                        RFIntHostApp.pF_StoreGatewayUID_PANID(pns->pkt.rmts.gatewayuid, pns->pkt.rmts.gatewaypanid, pns->pkt.rmts.gatewayaddrmode);
                        RFIntRFApp.pF_LookupSettingsAndConnectDeviceToNetwork();
                        g_pRFDeviceDescriptiveTable[index].COMLog.rxpacketremotecfg++;
                    }   else if (pns->pkt.rmts.flags & RF_REMOTE_NET_DEFAULT)   {
                        // check for gateway match

                        // accept new connecion parameters
                        RFIntHostApp.pF_StoreGatewayUID_PANID((UINT8*)GATEWAY_DEFAULT_UID, GATEWAY_DEFAULT_PANID, GATEWAY_ADDR_MODE);
                        RFIntRFApp.pF_LookupSettingsAndConnectDeviceToNetwork();
                        g_pRFDeviceDescriptiveTable[index].COMLog.rxpacketremotecfg++;
                    }
                }
    #endif
#endif
            break;

	case   RF_PACKET_DISCOVER_NETDEV_BEACON_REPLY:
		#ifndef  RF_NETWORK_MASTER_DEVICE
			if (isTraceEnabled)
				RFIntHostApp.pF_ConsolePrintf ("RF_PACKET_DISCOVER_NETDEV_BEACON_REPLY ok\r\n\r\n");
        //pass through to the next case 		
        #endif
		
        case    RF_PACKET_DISCOVER_NETDEV_BEACON:

#ifndef	RF_NETWORK_MASTER_DEVICE
		{	
            if (isTraceEnabled)
				RFIntHostApp.pF_ConsolePrintf ("RF_PACKET_DISCOVER_NETDEV_BEACON index=%02x\r\n", index);
			if( !g_roamingEnabled )
				break;
           
			if(RFDesc_GetIndexToRoamingDescriptiveTableByUID(g_pRFDeviceDescriptiveTable[index].IDInfo.uid,&indexInRoamingTable))
            {
				//note we are using this index to store LQ to the data field!!!
				//note: data[] has size of RF_DEVICE_DATA_HOLD_BUFFER_SIZE which is 258, 
				//      RF_GW_LQ_BUFFER_SIZE goes only from 0 to 256, the last space is used for other purpose.
				UINT16 LQ_dataIndex = g_pRFRoamingDescriptiveTable[indexInRoamingTable].RxBuf.index;
				if(g_pRFRoamingDescriptiveTable[indexInRoamingTable].linkeactiveestablishedtime)
				{
					g_pRFRoamingDescriptiveTable[indexInRoamingTable].LQ.linkqualityAtLocalDevice = \
					   g_pRFRoamingDescriptiveTable[indexInRoamingTable].RxBuf.data[LQ_dataIndex] = \
									  g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtLocalDevice;
	
					g_pRFRoamingDescriptiveTable[indexInRoamingTable].RxBuf.index++;
					g_pRFRoamingDescriptiveTable[indexInRoamingTable].RxBuf.index %= RF_GW_LQ_BUFFER_SIZE;
	
					g_pRFRoamingDescriptiveTable[indexInRoamingTable].linkeactiveestablishedtime = OS_RetrieveClock();
					g_pRFRoamingDescriptiveTable[indexInRoamingTable].rxsequence++;
				}
				else//fill out enough when power up
				{
					RFRecMsg_FillLQ_Buffer(indexInRoamingTable,g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtLocalDevice);
				}
			}
		}
#endif
            break;

        default:
            g_pRFDeviceDescriptiveTable[index].COMLog.rxpacketinvalidcmd++;
            //RFIntHostApp.pF_ErrorPrintf  ("RFRecMsg: Unexpected net svc packet received index=%02x, pkt type=%02x, pkt subtype=%02x, len=%02x\r\n", index, ppkthdr->type, pns->msgtype, ppkthdr->length);
            res = FAILURE;
            break;
    }

    return (res);
}
#ifndef	RF_NETWORK_MASTER_DEVICE
/**
 *
 * \author      John Cheng
 *
 * \brief		This function fills the buffer with LQ values
 *
 *
 * \detail
 *
 * \date        12/24/2008
 *
 * \param		UINT8 index, UINT8 linkQuality
 *
 * \return		void
 *
 * \retval		none
 */
void RFRecMsg_FillLQ_Buffer(UINT8 index, UINT8 linkQuality)
{
	INT8 i;
	g_pRFRoamingDescriptiveTable[index].LQ.linkqualityAtLocalDevice = linkQuality;

	g_pRFRoamingDescriptiveTable[index].RxBuf.index = 0;
	g_pRFRoamingDescriptiveTable[index].rxsequence  = 0;
	for(i=0; i<GW_RM_MAX_MV_AV_FILTER_SZ; i++)
	{
		g_pRFRoamingDescriptiveTable[index].RxBuf.data[i] = linkQuality;
		g_pRFRoamingDescriptiveTable[index].RxBuf.index++;
		g_pRFRoamingDescriptiveTable[index].rxsequence++;
	}
	g_pRFRoamingDescriptiveTable[index].linkeactiveestablishedtime = OS_RetrieveClock();
}
#endif
/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function processes all application packets not including the link request packet
 *
 *
 * \detail
 *
 * \date        10/13/2006
 *
 * \param       UINT8 index, UINT8 *pBuf, UINT16 Len, mcpsToNwkMessage_t *pMsgIn
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 */
UINT16 RFRecMsg_ProcessOtherDataPackets(UINT8 index, UINT8 *pBuf, UINT16 Len)   {
    UINT16 res = SUCCESS;
    TYPE_INTERNAL_RF_PACKET_HEADER  *ppkthdr = (TYPE_INTERNAL_RF_PACKET_HEADER *) pBuf;
    UINT8*  pdata                   = pBuf+sizeof(TYPE_INTERNAL_RF_PACKET_HEADER);
    UINT32  datalen                 = Len-sizeof(TYPE_INTERNAL_RF_PACKET_HEADER);
    BOOL    isTraceEnabled;

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

    switch (ppkthdr->type)  {
#ifdef  RF_NETWORK_MASTER_DEVICE
        case    RF_PACKET_ENTER_SLEEP_STATE:
            //  Switch device link state from active to sleep
            RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_SLEEP);
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Sleep Packet received index=%02x\r\n", index);
            }
            break;

        case    RF_PACKET_ENTER_POWERDOWN_STATE:
            //  Switch device link state fronm active to power down
            RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_POWERDOWN);
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Power down Packet received index=%02x\r\n", index);
            }
            break;

        case    RF_PACKET_HEART_BEAT:
            RFRecMsg_ProcessHeartBeatPacket (index, (TYPE_RF_HEART_BEAT_PACKET_DATA*)pdata);
            break;

        case    RF_PACKET_COMMUNICATION_ERROR:
            RFRecMsg_ProcessCommunicationErrorPacket (index, (TYPE_RF_COMMUNICATION_ERROR_PACKET_DATA*) pdata);
            break;

        case    RF_PACKET_DEVICE_DATA_FIELD_REPLY:
            RFRecMsg_SubmitPacketToRFApplication (index, pBuf, Len);
            break;
#else
        case    RF_PACKET_CTLSYS_CONNECT_STATUS:
            RFRecMsg_ProcessControlSysConnectPacket (index, (TYPE_RF_CS_STATUS_PACKET_DATA*)pdata);
            break;

        case    RF_PACKET_BUFFER_OVERFLOW_REPORT:
            RFRecMsg_ProcessBufferOverFlowPacket (index, (TYPE_RF_BUFFER_OVERFLOW_PACKET_DATA*)pdata);
            break;

        case    RF_PACKET_COMMUNICATION_ERROR:
            RFRecMsg_ProcessCommunicationErrorPacket (index, (TYPE_RF_COMMUNICATION_ERROR_PACKET_DATA*) pdata);
            break;

        case    RF_PACKET_LINK_REPLY:
            RFRecMsg_SubmitPacketToRFApplication (index, pBuf, Len);
            break;

        case    RF_PACKET_GATEWAY_RESTART:
            RFRecMsg_ProcessGatewayReStartPacket (index, (TYPE_RF_GATEWAY_RESTART_PACKET_DATA*) pdata);
            break;

        case    RF_PACKET_DEVICE_DATA_FIELD_REQUEST:
            RFRecMsg_SubmitPacketToRFApplication (index, pBuf, Len);
            break;
#endif

        case    RF_PACKET_ECHO_REQUEST:
            RFRecMsg_ProcessEchoRequestPacket (index, (TYPE_RF_ECHO_REQUEST_PACKET_DATA*) pdata, datalen);
            break;

        case    RF_PACKET_LOCAL_CRESNET:
            RFRecMsg_ProcessLocalCresnetPackets (index, pdata, datalen);
            break;

        case    RF_PACKET_ECHO_REPLY:
            RFRecMsg_ProcessEchoReplyPacket (index, pBuf, Len);
            break;

        case    RF_PACKET_DEBUG1_INFO_REQUEST:
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Debug1 request Packet received index=%02x\r\n", index);
            }
            break;

        case    RF_PACKET_DEBUG1_INFO_REPLY:
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Debug1 reply Packet received index=%02x\r\n", index);
            }
            break;

        case    RF_PACKET_DEBUG2_INFO_REQUEST:
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Debug2 request Packet received index=%02x\r\n", index);
            }
            break;

        case    RF_PACKET_DEBUG2_INFO_REPLY:
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Debug2 reply Packet received index=%02x\r\n", index);
            }
            break;

        case    RF_PACKET_THROUGHPUT_TST:
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Through-put test packet received index=%02x\r\n", index);
            }
            break;

        case    RF_PACKET_NETWORK_SERVICES:
            RFRecMsg_ProcessNetworkServicesPacket (index, pBuf, Len);
            break;

        case    RF_PACKET_DISCARD:
            // unused pkt 
            break;

        default:
            g_pRFDeviceDescriptiveTable[index].COMLog.rxpacketinvalidcmd++;
            RFIntHostApp.pF_ErrorPrintf  ("RFRecMsg: Unexpected pkt recved index=%02x, type=%02x, len=%02x, data [%02x%02x%02x%02x%02x%02x]\r\n", index, ppkthdr->type,
                                                                      ppkthdr->length,
                                                                      *pBuf, *(pBuf+1),
                                                                      *(pBuf+2),*(pBuf+3),
                                                                      *(pBuf+4),*(pBuf+5));
            res = FAILURE;
            break;
    }

    return (res);
}

#ifdef  RF_NETWORK_MASTER_DEVICE
/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function process the link request packet and other packets from devices that have not been linked yet
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT8 index, UINT8 *pBuf
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 */
UINT16 RFRecMsg_ProcessLinkRequest(UINT8 index, UINT8 *pBuf)    {
    TYPE_INTERNAL_RF_PACKET_HEADER   *ppkthdr = (TYPE_INTERNAL_RF_PACKET_HEADER *) pBuf;
    TYPE_RF_LINK_REQUEST_PACKET_DATA *pdata = (TYPE_RF_LINK_REQUEST_PACKET_DATA *) (pBuf+sizeof(TYPE_INTERNAL_RF_PACKET_HEADER));
    TYPE_RF_DESCRIPTIVE_TABLE *pdst         = (g_pRFDeviceDescriptiveTable+index);
    UINT16  acceptconnection = FALSE, reason;
    //UINT8   rfid;
    INT8    name[GATEWAY_NAME_MAX_SIZE];
    BOOL    replyposted = FALSE;
    BOOL    isTraceEnabled;
    INT8    buf[50];

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

	HPRF_NTOHS(pdata->panid);
	HPRF_NTOHL(pdata->tsid);

    // MNT - 5/23/2007 -  Since this is only needed by the master device
#ifdef  RF_NETWORK_MASTER_DEVICE
    GetHostname(name, sizeof(name));
#endif

    if          (ppkthdr->type != RF_PACKET_LINK_REQUEST)   {
        //  unexpected packet waiting for link request packet. Reject link request  (link was lost either the device went out of range etc..)
        if (SUCCESS == RFTransmit_BuildLinkReplyPacketAndPostToPipeByIndex (index, RF_PACKET_LINK_REJECTED_DEVICE_NOT_LINKED, 0x0000, name))    {
            replyposted = TRUE;
        }

        //  log error locally
        if (isTraceEnabled) {
            ConvertLinkStatetoString (pdst->linkstate, buf);
            RFIntHostApp.pF_ConsolePrintf ("Device not linked, needs link request packet, link state %s, index[%02x]\r\n", buf, index);
        }
    }   else {
        if (SUCCESS == RFDesc_IsDuplicateUID(index, pdata->tsid))   {
            //if (pdata->linkcondition != RF_PACKET_LINK_ONLY_IF_GATEWAY_IN_ACQUIRE)    {
                //  duplicate uid, reject link request
                //if (SUCCESS == RFTransmit_BuildLinkReplyPacketAndPostToPipeByIndex (index, RF_PACKET_LINK_REJECTED_DUPLICATE_UID, pdata->randomno, name)) {
                //  replyposted = TRUE;
                //}
                //  7/5/07 - HS (accept devices with duplicate UID/TSID, log error - otherwise, this device will not be
                //               linked and gateway reset is required.
                //               1- A duplicate TSID error will usually occur if the device is assigned a new TSID and was
                //                  previously connected to the gateway.
                //               2- Devices with duplicate UID will disconnect and reconnect
            //}
            RFIntHostApp.pF_ErrorPrintf ("Link request packet error, duplicate UID[%s], tsid[%04x], rfid[%02x] \r\n", ConvertUIDtoString ((UINT8*)&pdst->IDInfo.uid, buf, gAddrModeLong_c), pdata->tsid, pdata->rfid);
        }

        if          (pdata->linkcondition == RF_PACKET_LINK_ONLY_IF_GATEWAY_IN_ACQUIRE) {
			if (RFAPP_AcquireStatus())  {
                //  Only respond to the link request if in acquire
                acceptconnection = TRUE;
            }   else    {
                //  Not in acquire - do not respond to this message
                if (isTraceEnabled) {
                    RFIntHostApp.pF_ConsolePrintf ("Link request packet, no response sent not in acquire mode, RFID %02X\r\n", pdst->IDInfo.rfid);
                }
            }
        }   else if (pdata->linkcondition == RF_PACKET_LINK_UNCONDITIONAL)  {
            //  Add device to the list
            acceptconnection = TRUE;
        }   else if (pdata->linkcondition == RF_PACKET_LINK_QUERY)  {
            //  request link info
            if (SUCCESS == RFTransmit_BuildLinkReplyPacketAndPostToPipeByIndex (index, RF_PACKET_LINK_QUERY_REPLY, pdata->randomno, name))  {
                replyposted = TRUE;
            }
        }   else    {
            //  Unknown condition, reject device
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Link request packet received with unknown reason, RFID %02X\r\n", pdst->IDInfo.rfid);
            }
            acceptconnection = FALSE;
            if (SUCCESS == RFTransmit_BuildLinkReplyPacketAndPostToPipeByIndex (index, RF_PACKET_LINK_REJECTED_UNKNOWN_LINK_CONDITION, pdata->randomno, name))  {
                replyposted = TRUE;
            }
        }

        if (acceptconnection)   {
            if (pdst->linkstate == RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS)   {
                reason = RF_PACKET_LINK_SUCCESSFUL_NEW_ENTRY;
                if (isTraceEnabled) {
                    RFIntHostApp.pF_ConsolePrintf ("Link request packet accepted new entry, descriptive entry %02X, RFID %02X\r\n", index, pdata->rfid);
                }
            }   else    {
                reason = RF_PACKET_LINK_SUCCESSFUL_EXISTING_ENTRY;
                if (isTraceEnabled) {
                    RFIntHostApp.pF_ConsolePrintf ("Link request packet accepted existing entry, descriptive entry %02X, RFID %02X\r\n", index, pdata->rfid);
                }
            }

            RFDesc_UpdateDeviceIDInfo(index, pdata->power, pdata->tsid, pdata->rfid, pdata->panid, pdata->deviceoptions, (UINT8*)&pdata->idstring, (ppkthdr->length - (sizeof(TYPE_INTERNAL_RF_PACKET_HEADER)-sizeof(ppkthdr->length)+((UINT32)pdata->idstring-(UINT32)pdata))));
            //  if link request reason is acquire mode then donot make connection active yet
            if (pdata->linkcondition != RF_PACKET_LINK_ONLY_IF_GATEWAY_IN_ACQUIRE)  {
                if (pdst->linkstate == RF_DEVICE_LINK_STATE_ACTIVE) {
                    RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_FAIL);
                }
                RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_ACTIVE);
            }

            //  accept link request
            if (SUCCESS == RFTransmit_BuildLinkReplyPacketAndPostToPipeByIndex (index, (UINT8)reason, pdata->randomno, name))  {
                replyposted = TRUE;
            }

#ifdef  RF_NETWORK_MASTER_DEVICE
            //  gateway specific
            //  check and show info string on the console of devices that were just acquired
            if (pdata->linkcondition != RF_PACKET_LINK_ONLY_IF_GATEWAY_IN_ACQUIRE)  {
                if (RFAPP_AcquireStatus())  {
                    if (pdata->flags & ACQUIRE_SUCCESSFUL)  {
                        ShowDeviceInfoInAcquireProcess (index, TRUE);
                        //RFIntHostApp.pF_ConsolePrintf ("Link request packet accepted existing entry, descriptive entry %02X, RFID %02X\r\n", index, pdata->rfid);
                    }   else    {
                        ShowDeviceInfoInAcquireProcess (index, FALSE);
                    }
                }
            }
#endif
        }
    }

    //  only allowed to release TEMP_CONN descriptive entries if msg was not processed for whatever reason
    if ((!replyposted) && (!acceptconnection) && (pdst->linkstate == RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS))    {
        //  set the descriptive entry to empty state
        RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_EMPTY);
    }

    return ((acceptconnection||replyposted)?SUCCESS:FAILURE);
}
#endif //#ifdef  RF_NETWORK_MASTER_DEVICE

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function processes incoming MCP data packets. The segmented packets are merged and then forwarded to the
 *              application applicaiton for processing. Link request and other packets unrecognized devices are handed over
 *              to the link request process.
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT8 index, UINT8 *pBuf, UINT8 Len
 * \param       UINT8 index, mcpsToNwkMessage_t *pMsgIn
 *
 *
 * \return      UINT16 res
 *
 * \retval      SUCCESS/FAILURE
 */
UINT16 RFRecMsg_ProcessMessagesInDataPacket (UINT8 index, UINT8 *pBuf, UINT8 Len)   {
    TYPE_EXTERNAL_RF_PACKET_HEADER *ppkthdr = (TYPE_EXTERNAL_RF_PACKET_HEADER *) pBuf;
    TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
    UINT16  process = FALSE, error = RF_COMM_ERR_PACKET_NONE;
    const TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;
    //UINT8   i;
    BOOL    isTraceEnabled;
    UINT32  intdatalen;
    UINT32  extdatalen;
    UINT16  res = SUCCESS;
#ifdef RF_NETWORK_MASTER_DEVICE
    BOOL    lkreqprocessed = FALSE;
#endif

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

    //  The first segment number is 1
    if (pdst->RxBuf.pkthdr.segmentno < 1)   {   pdst->RxBuf.pkthdr.segmentno = 1;   }

    while   (Len)   {
        process = FALSE;
        if ((ppkthdr->length<(sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER)-sizeof(ppkthdr->length))) 
			|| (ppkthdr->length>RF_DEVICE_TRANSMIT_BUFFER_SIZE)
			|| (ppkthdr->length+sizeof(ppkthdr->length) > Len ))  {
				RETAILMSG(1, (TEXT("Invalid HPRF Packet size: %i\r\n"),ppkthdr->length));
		    //  invalid sub-packet length, discard
            pdst->COMLog.rxpacketinvalidcmd++;
            pdst->RxBuf.pkthdr.segmentno     = 1;
            pdst->RxBuf.pkthdr.totalsegments = 1;
            //  inform device about rx packet error 
            RFTransmit_BuildCommunicationErrorPacketAndPostToPipeByIndex (index, RF_COMM_ERR_SEGMENT_LEN_SHORT, pdst->rxsequence-1, pdst->rxsequence-1, pdst->RxBuf.pkthdr.totalsegments, pdst->RxBuf.pkthdr.segmentno, ppkthdr->segmentno);
            //process = FALSE;
            Len = 0;
            //  terminate the while loop, discard the rest of the packet
            break;
        }

        //  Trigger Low priority message posting delay whenever any packet is received except the heart beat packet
        if (ppkthdr->type != RF_PACKET_HEART_BEAT)  {
            RFTransmit_ReTriggerLowPriorityMessagePostDelay ();
        }

        if (ppkthdr->totalsegments > 1) {

            extdatalen = (ppkthdr->length+sizeof(ppkthdr->length))-sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER);

            //  Process incoming segmented packet
            if (pdst->RxBuf.segmenterror && (ppkthdr->segmentno != 1))  {
                //  Error case:
                //  Once an error occurs in a segmented packet either because, it is out of seq or missing pieces
                //  stay in this state unitl either a non-segmented packet is received or the first segment of another segmenetd
                //  packet is received. Also, discard this packet
                //
                //  In some instances there could be two or more errors generated in case the packet segments are out of alignment
                //  and the segment no continues to be 1. The reason for this is that the receive needs to be able to handle consective
                //  segmented packets in case the first segmented packet is out of sequence or has missing piece and the 2nd segmented packets
                //  are ok.
                //  This mechanism will allow the first segment of packets to stored but, will not execute them if out of sequence.
                //  The likely hood of having segmenetd packets contiounsly with segment 1 is not high, it is more likely that segmented
                //  could be missing in which case the transmitting device will be informed. It could retransmit the packet.
            }   else    {
                if (ppkthdr->segmentno == pdst->RxBuf.pkthdr.segmentno) {
                    if (ppkthdr->segmentno == 1) {
                        //  First segment, clear error flag
                        pdst->RxBuf.segmenterror = FALSE;
                        //  Copy packet header
                        pdst->RxBuf.pkthdr.length = sizeof(TYPE_INTERNAL_RF_PACKET_HEADER)-sizeof(pdst->RxBuf.pkthdr.length);
                        memcpy ((UINT8*)&pdst->RxBuf.pkthdr.type, (UINT8*)&ppkthdr->type, pdst->RxBuf.pkthdr.length);

                        if (pdst->RxBuf.pkthdr.totalsegments != pdst->RxBuf.pkthdr.segmentno)   {
                            pdst->COMLog.rxpacketsegmented += 1;
                        }
                    }

                    if ((pdst->RxBuf.pkthdr.length+sizeof(pdst->RxBuf.pkthdr.length)-sizeof(TYPE_INTERNAL_RF_PACKET_HEADER)+extdatalen) < sizeof(pdst->RxBuf.data)) {
                        //  Copy packet data
                        memcpy ((UINT8*)&pdst->RxBuf.data[(pdst->RxBuf.pkthdr.length+sizeof(pdst->RxBuf.pkthdr.length))-sizeof(TYPE_INTERNAL_RF_PACKET_HEADER)], (UINT8*) (pBuf+sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER)), extdatalen);
                        pdst->RxBuf.pkthdr.length += extdatalen;

                        //  check if all segments received
                        if (pdst->RxBuf.pkthdr.totalsegments == pdst->RxBuf.pkthdr.segmentno)   {
                            //  All segments were received, packet is complete, submit for processing
                            pdst->RxBuf.pkthdr.segmentno     = 1;
                            pdst->RxBuf.pkthdr.totalsegments = 1;
                            process = TRUE;                         
                        }   else    {
                            //  Prepare for next segment reception (cannot process yet more data to come)
                            pdst->RxBuf.pkthdr.segmentno += 1;
                        }
                    }   else    {
                        //  error, incoming packet is bigger than the buffer size
                        error = RF_COMM_ERR_SEGMENT_LEN_TOO_BIG;
                        pdst->RxBuf.segmenterror = TRUE;
                        RFIntHostApp.pF_ErrorPrintf ("Received segmented len too large: error %02X, RFID %02x\r\n", error, pdst->IDInfo.rfid);
                    }
                }   else    {
                    //  error, segmented packet is out of order: send message to device
                    error = RF_COMM_ERR_SEGMENT_OUT_OF_SEQUENCE;
                    pdst->RxBuf.segmenterror = TRUE;
                    if (isTraceEnabled) {
                        RFIntHostApp.pF_ConsolePrintf ("Received segment out of sequence: error %02X, RFID %02x\r\n", error, pdst->IDInfo.rfid);
                    }
                }
            }
        }   else    {
            //  process non-segmented packet

            //  Check for error case: Was in the middle of receving a segmented packet but, a non-segmented packet was
            //  received and all pieces of the segmented packets were not received set the error flag
            if ((pdst->RxBuf.pkthdr.totalsegments > 1) && (!pdst->RxBuf.segmenterror))  {
                //  Error occured in segment reception, all segments were not received
                error = RF_COMM_ERR_SEGMENT_MISSING;
                if (isTraceEnabled) {
                    RFIntHostApp.pF_ConsolePrintf ("Received segment missing segments: error %02X, RFID %02x\r\n", error, pdst->IDInfo.rfid);
                }
            }

            //  clear segmented packet error
            pdst->RxBuf.segmenterror = FALSE;

            //  copy packet header and data
            memcpy ((UINT8*)&pdst->RxBuf.pkthdr.type, (UINT8*)&ppkthdr->type, ppkthdr->length);
            pdst->RxBuf.pkthdr.length = ppkthdr->length;
            process    = TRUE;          
        }

        if (error != RF_COMM_ERR_PACKET_NONE)   {
            //  Inform device about the segment error
            RFTransmit_BuildCommunicationErrorPacketAndPostToPipeByIndex (index, error, pdst->rxsequence-1, pdst->rxsequence-1, pdst->RxBuf.pkthdr.totalsegments, pdst->RxBuf.pkthdr.segmentno, ppkthdr->segmentno);
            pdst->RxBuf.pkthdr.segmentno     = 1;
            pdst->RxBuf.pkthdr.totalsegments = 1;
            pdst->COMLog.rxpacketsegmentationerrcnt += 1;
            error   = RF_COMM_ERR_PACKET_NONE;
            process = FALSE;
            Len = 0;
            //  terminate the while loop, discard the rest of the packet
            break;
        }

        if (process)    {
            //  update concatenated packet count
            pdst->COMLog.rxpacketconcatcount += 1;

            intdatalen = (pdst->RxBuf.pkthdr.length+sizeof(pdst->RxBuf.pkthdr.length))-sizeof(TYPE_INTERNAL_RF_PACKET_HEADER);
#ifdef      RF_NETWORK_MASTER_DEVICE
            //  (6/8/07 - HS) modified all FAIL,SLEEP,POWER-DOWN and TEMP-CONN devices are required to issue LinkRequest prior to connection
            //  (7/12/07 - HS) do not pass packets from other gateways into the link process
            //  (9/21/07) - HS) to avoid sending multiple linkreq reject messages in a concatenated packet, linkreq handler 
            //                  is called once per packet   
            //if ((pdst->linkstate == RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS) || (pdst->RxBuf.pkthdr.type == RF_PACKET_LINK_REQUEST))    {
            if (((pdst->linkstate != RF_DEVICE_LINK_STATE_ACTIVE) || (pdst->RxBuf.pkthdr.type == RF_PACKET_LINK_REQUEST)) &&
                (pdst->RxBuf.pkthdr.type != RF_PACKET_NETWORK_SERVICES))    {        
                if (ppkthdr->options & RF_DEVICE_TYPE_GATEWAY)  {
                    //  pkt from another gw, not processed  
                    Len = 0;    
                    //  release descriptive tbl 
                    if (pdst->linkstate == RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS)   {
                        RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_EMPTY);         
                    }
                }   else    {
                    if (!lkreqprocessed)    {
                        if (SUCCESS != RFRecMsg_ProcessLinkRequest(index, (UINT8*)&pdst->RxBuf.pkthdr)) {
                            //  Discard the rest of the packet, link was not accepted
                            Len = 0;
                            if (isTraceEnabled) {
                                RFIntHostApp.pF_ConsolePrintf ("Received packets with link request error or link request rejected (link request packet failed) RFID %02x \r\n", pdst->IDInfo.rfid);
                            }
                        }   else    {
                            lkreqprocessed = TRUE;
                        }
                    }
                        
                    //  forward all cresnet packets for unlinked devices to the control system 
                    if (Len && (pdst->RxBuf.pkthdr.type == RF_PACKET_CRESNET))  {
                        RFIntHostApp.pF_ProcessCresnetPacket (pdst->RxBuf.data, intdatalen);    
                    }
                }
            }   else if (pdst->RxBuf.pkthdr.type == RF_PACKET_CRESNET)  {
#else
            if  (pdst->RxBuf.pkthdr.type == RF_PACKET_CRESNET)  {
#endif
            
#ifdef      RF_NETWORK_MASTER_DEVICE
                if (IsSlaveDevicePassToActive())    {
                    if ((pdst->RxBuf.data[2] == CNET_MULTISERIAL) && (pdst->RxBuf.data[3] == 0x00)) {                       
                        //  in pass to mode dump packet to the console                  
                        RFIntHostApp.pF_ConsolePrintf("%s", pdst->RxBuf.data[4]);
                    }
                }   else    {
                    //  Submit Cresnet packet for processing, strip away the packet header submit only the CRESNET packet header and data
                    RFIntHostApp.pF_ProcessCresnetPacket (pdst->RxBuf.data, intdatalen);
                }
#else
                RFRecMsg_ProcessOptions (ppkthdr->options);
                //  submit packet to the host
                (pHF->pF_ProcessCresnetPacket) (pdst->RxBuf.data, intdatalen);
#endif
            }   else    {
                //  Submit other packets for processing
                res = RFRecMsg_ProcessOtherDataPackets(index, (UINT8*)&pdst->RxBuf.pkthdr,
                                                        pdst->RxBuf.pkthdr.length+sizeof(pdst->RxBuf.pkthdr.length));
                if ((res == SUCCESS) && (ppkthdr->type != RF_PACKET_NETWORK_SERVICES))  {
                    RFRecMsg_ProcessOptions (ppkthdr->options);
                }
            }
        }

        if (Len)    {
            //  adjust the buffer to point to the next packet
            ppkthdr = (TYPE_EXTERNAL_RF_PACKET_HEADER *) pBuf;
            Len    -= (ppkthdr->length+sizeof(ppkthdr->length));
            pBuf   += (ppkthdr->length+sizeof(ppkthdr->length));
            ppkthdr = (TYPE_EXTERNAL_RF_PACKET_HEADER *) pBuf;
        }
    }

    return (process?res:FAILURE);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function checks the incoming packet integrity and determines whether it should processed further
 *
 * \detail
 * \date        10/13/2006
 *
 * \param       UINT8 *pindex, mcpsToNwkMessage_t *pMsgIn, BOOL broadcast
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 */
UINT16 RFRecMsg_QualifyDataPacket (UINT8 *pindex, mcpsToNwkMessage_t *pMsgIn, BOOL broadcast)   {
    UINT8   index = 0;
    TYPE_RF_PAYLOAD_HEADER *payldhdr = (TYPE_RF_PAYLOAD_HEADER *) &pMsgIn->msgData.dataInd.msdu;
    TYPE_EXTERNAL_RF_PACKET_HEADER *pkthdr   = (TYPE_EXTERNAL_RF_PACKET_HEADER*)&pMsgIn->msgData.dataInd.msdu[sizeof(TYPE_RF_PAYLOAD_HEADER)];
    TYPE_RF_DESCRIPTIVE_TABLE *pdst  = g_pRFDeviceDescriptiveTable;
    UINT8   process = FALSE;
    //TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHoldBuf;
    INT8    buf[200], buf1[30];
    BOOL    isTraceEnabled;
    UINT8   bdrindex;
    BOOL    newEntry = FALSE;

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

    //  Check to make sure that the packet contains at least one packet of payload data
    if  (   (pMsgIn->msgData.dataInd.msduLength >= (sizeof(TYPE_RF_PAYLOAD_HEADER)+sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER))) && (pkthdr->length >= sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER)) &&
            (pMsgIn->msgData.dataInd.msduLength < 127)) {
		HPRF_NTOHS(payldhdr->signature);
        if (payldhdr->signature == HPRFGW_PACKET_SIGNATURE) {
            //  for network services packets uses the dedicated network services descriptive table entry
            if (pkthdr->type == RF_PACKET_NETWORK_SERVICES) {
                if (SUCCESS != RFDesc_GetNetworkServiceDescriptiveTableEntryByUID (&index, pMsgIn->msgData.dataInd.srcAddr, ConvUINTToBigIndian(pMsgIn->msgData.dataInd.srcPanId), pMsgIn->msgData.dataInd.srcAddrMode)) {
					if (SUCCESS == RFDesc_CreateNetworkServiceDescriptiveTableEntryByUID (&index, pMsgIn->msgData.dataInd.srcAddr, ConvUINTToBigIndian(pMsgIn->msgData.dataInd.srcPanId), pMsgIn->msgData.dataInd.srcAddrMode, payldhdr->sequence))    {
                        process = TRUE;
                        newEntry = TRUE;
                     }
                }   else    {
                    process = TRUE;
                }
            }   else if (RFDesc_GetIndexToDescriptiveTableByUID (pMsgIn->msgData.dataInd.srcAddr, &index))  {
                //  Check if the UID was found in the descriptive table, if entry is not found create one
                process = TRUE;
            }   else    {
            #ifndef	RF_NETWORK_MASTER_DEVICE
                index = -1;
				if( g_roamingEnabled && RFAPP_AcquireStatus())
                {
					if((SUCCESS != RFDesc_CreateEntryInRoamingDescriptiveTable(&index, pMsgIn->msgData.dataInd.srcAddr, ConvUINTToBigIndian(pMsgIn->msgData.dataInd.srcPanId), pMsgIn->msgData.dataInd.srcAddrMode, payldhdr->sequence)))
					{
						return FAILURE;
					}
					else
					{
						g_pRFRoamingDescriptiveTable[index].LQ.linkqualityAtLocalDevice = pMsgIn->msgData.dataInd.mpduLinkQuality;
						g_pRFRoamingDescriptiveTable[index].LQ.linkqualityAtRemoteDevice = payldhdr->linkquality;
					}
				}
             #endif
                if (SUCCESS == RFDesc_CreateEntryInDescriptiveTable (&index, pMsgIn->msgData.dataInd.srcAddr, ConvUINTToBigIndian(pMsgIn->msgData.dataInd.srcPanId), pMsgIn->msgData.dataInd.srcAddrMode, payldhdr->sequence))  {
                    //  entry was created successfully, rx sequence count was syncronized to pass the sequence check below
                    //TODO: revert
                    //if (isTraceEnabled) {
                    if(1) {
                        RFIntHostApp.pF_ConsolePrintf ("Descriptor table was successfully created at index %02x, rfid %02x\r\n", index, g_pRFDeviceDescriptiveTable[index].IDInfo.rfid);
                    }
                    newEntry = TRUE;                
                    process = TRUE;
                }   else    {
                    //  Descriptor table is full, packet cannot be processed
                    sprintf (buf, "RF Descriptor entry table full: Device[%s] was not linked\r\n", ConvertUIDtoString ((UINT8*) &pMsgIn->msgData.dataInd.srcAddr, buf1, gAddrModeLong_c));
                    RFIntHostApp.pF_ErrorHandler(buf);
                    process = FALSE;
                    if (isTraceEnabled) {
                        RFIntHostApp.pF_ConsolePrintf(buf);
                    }
					#ifndef	RF_NETWORK_MASTER_DEVICE
					if(g_roamingEnabled && RFDesc_GetIndexToRoamingDescriptiveTableByUID(pMsgIn->msgData.dataInd.srcAddr,&bdrindex)) 
					{
						g_pRFRoamingDescriptiveTable[bdrindex].COMLog.rxbufoverflowcount += 1;
						if (isTraceEnabled)
							RFIntHostApp.pF_ConsolePrintf("\r\nGateway entry index =%d\r\n",bdrindex);
					}
					#endif 
                }
            }
        }   else    {
            //  packet signature invalid
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Received packet containing invalid signature from uid[%s] panid[%02x%02x] addrmode[%x] len[%02x]\r\n",
                            ConvertUIDtoString((UINT8*)(&pMsgIn->msgData.dataInd.dstAddr), buf, pMsgIn->msgData.dataInd.dstAddrMode),
                            pMsgIn->msgData.dataInd.dstPanId[0], pMsgIn->msgData.dataInd.dstPanId[1], pMsgIn->msgData.dataInd.dstAddrMode,
                            pMsgIn->msgData.dataInd.msduLength);
            }
            pdst->COMLog.rxpackethdrftrerror++;
        }
    }   else    {
        //  packet length invalid
        if (isTraceEnabled) {
            RFIntHostApp.pF_ConsolePrintf ("Received invalid len packet uid[%s] panid[%02x%02x] addrmode[%x] len[%02x]\r\n",
                    ConvertUIDtoString((UINT8*)(&pMsgIn->msgData.dataInd.dstAddr), buf, pMsgIn->msgData.dataInd.dstAddrMode),
                    pMsgIn->msgData.dataInd.dstPanId[0], pMsgIn->msgData.dataInd.dstPanId[1], pMsgIn->msgData.dataInd.dstAddrMode,
                    pMsgIn->msgData.dataInd.msduLength);
        }
        pdst->COMLog.rxpacketlenerror++;
        process = FALSE;
    }

    if (process)    {
        process = FALSE; 
        pdst = pdst+index;
        RFHeartBeat_ReTriggerTimer (index);
        //  store local/remote device link quality
        RFDesc_UpdateLinkQualityOfLocalDevice  (index, pMsgIn->msgData.dataInd.mpduLinkQuality);
        RFDesc_UpdateLinkQualityOfRemoteDevice (index, payldhdr->linkquality);
        //  update total number of packet count received from this device
        pdst->COMLog.rxpacketcount += 1;
        pdst->COMLog.rxpacketpayloadbytecount += pMsgIn->msgData.dataInd.msduLength;

        //if (pdst->linkstate == RF_DEVICE_LINK_STATE_DISCONNECTED) {
        //  //  Packet received for a device whose link state is disconnect (device should issue link request first)
        //  sprintf (buf, "Received packet from disconnected device:  index[%02X] rfid[%02x]\r\n", index, pdst->IDInfo.rfid);
        //  RFIntHostApp.pF_ErrorHandler(buf);
        //} else if (broadcast) {
        if (broadcast)  {
            if          (SUCCESS == RFDesc_FindBroadcastDeviceUID (&bdrindex, pMsgIn->msgData.dataInd.srcAddr, ConvUINTToBigIndian(pMsgIn->msgData.dataInd.srcPanId)))  {
				//  throw-out repeat broadcasts
                if ((BDR.rx[bdrindex].sequence-1) != payldhdr->sequence)    {
                    BDR.rx[bdrindex].sequence = payldhdr->sequence; BDR.rx[bdrindex].sequence++;
                    process = TRUE;
                }
            }   else if (SUCCESS == RFDesc_AddBroadcastDeviceUID (&bdrindex, pMsgIn->msgData.dataInd.srcAddr, ConvUINTToBigIndian(pMsgIn->msgData.dataInd.srcPanId), payldhdr->sequence))   {
				BDR.rx[bdrindex].sequence = payldhdr->sequence; BDR.rx[bdrindex].sequence++;
                process = TRUE; 
            }   
        #ifndef	RF_NETWORK_MASTER_DEVICE
                else if(g_roamingEnabled && RFDesc_GetIndexToRoamingDescriptiveTableByUID(pMsgIn->msgData.dataInd.srcAddr,&bdrindex)) {
				process = TRUE;	
			}
        #endif    
        }   else if (pdst->rxsequence == payldhdr->sequence)    {
            //  sequence numbers are same 
			process = TRUE;
        }   else if ((pdst->linkstate == RF_DEVICE_LINK_STATE_SLEEP) || (pdst->linkstate == RF_DEVICE_LINK_STATE_FAIL) || (pdst->linkstate == RF_DEVICE_LINK_STATE_POWERDOWN))  { 
            //  (6/8/07 - HS) modified all FAIL,SLEEP,POWER-DOWN and TEMP-CONN devices are required to issue LinkRequest prior to connection
            //                the device is set to active after link request packet is received
            //  RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_ACTIVE);
            //  Syncronize sequence number
            pdst->rxsequence = payldhdr->sequence;
            process = TRUE;
        }   else if ((pkthdr->type == RF_PACKET_LINK_REQUEST) || (pkthdr->type == RF_PACKET_LINK_REPLY))    {
            //  For link request and reply packets ignore sequence count errors, these packets are used to establish initial connections
            pdst->rxsequence = payldhdr->sequence;
            process = TRUE;
        }   else if ((pdst->rxsequence-1) == payldhdr->sequence)    {
            //  duplicate packet, discard duplicate packet
            if (isTraceEnabled) {
				RFIntHostApp.pF_ConsolePrintf ("Received packet duplicate from, index[%02X], rfid[%02x]\r\n", index, pdst->IDInfo.rfid);
            }
            //  update duplicate packet count
            pdst->COMLog.rxpacketrepeatcount += 1;
        }   else    {
            //  Error: Rx sequence number does not match received packet sequence number
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Received packet out of sequence, index[%02X] rfid[%02x] rx seq[%02x] expected seq[%02x]\r\n", index, pdst->IDInfo.rfid, payldhdr->sequence, pdst->rxsequence);
            }
            //  issue packt out of sequence error on non-broadcast packets only
            RFTransmit_BuildCommunicationErrorPacketAndPostToPipeByIndex (index, RF_COMM_ERR_PACKET_OUT_OF_SEQUENCE, pdst->rxsequence, payldhdr->sequence, 0, 0, 0);
            //  update seq error counts
            pdst->COMLog.rxpacketsequenceErrcount += 1;
            //  syncronize the sequence number
            pdst->rxsequence = payldhdr->sequence;

            process = TRUE;
        }

        if (process)    {   
            //  Update sequence number for the next packet
            pdst->rxsequence++;
        }   else    {
            //  release descriptive table entry (this condition may occur because, repeat broadcast packets are ignored)
            if (newEntry && (pdst->linkstate == RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS)) {
                RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_EMPTY);         
            }
        }
    }

    *pindex = index;

    return ((process)?SUCCESS:FAILURE);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function is handed over the incoming MCP data packet, the packet addressing mode is checked,
 *              broadcast packets are checked for, the packet is passed on for further integrity checks and if okay then
 *              proccessed
 *
 * \detail
 *
 * \date        10/11/2006
 *
 * \param       mcpsToNwkMessage_t *pMsgIn
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 */
UINT16 RFRecMsg_DataPacketProcessing (mcpsToNwkMessage_t *pMsgIn)   {
    UINT16  result = FAILURE;
    UINT8   index;
    INT8    buf[100];
    BOOL    process = FALSE;
    BOOL    isTraceEnabled = FALSE;
    UINT16  dstpanid     = ConvUINTToBigIndian ((UINT8*)&pMsgIn->msgData.dataInd.dstPanId);
#ifdef IEEE_STD_BROADCAST
    UINT16  dstshortaddr = ConvUINTToBigIndian ((UINT8*)&pMsgIn->msgData.dataInd.dstAddr);
#endif
    UINT8   dstuid[UID_MAX_SIZE];
    BOOL    srcuidvalid;
    UINT8   srcuid[UID_MAX_SIZE];
    UINT16  srcpanid;
    UINT8   srcaddrmode;
    BOOL    broadcast = FALSE;
    TYPE_EXTERNAL_RF_PACKET_HEADER *pkthdr = (TYPE_EXTERNAL_RF_PACKET_HEADER*)&pMsgIn->msgData.dataInd.msdu[sizeof(TYPE_RF_PAYLOAD_HEADER)];

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

    if (pMsgIn->msgData.dataInd.srcAddrMode != gAddrModeLong_c) {
        if (isTraceEnabled) {
            //  Unacceptable source addressing mode
            RFIntHostApp.pF_ConsolePrintf ("MCP packet error, invalid src addr: uid[%s] panid[%04x] addrmode[%x]\r\n",  ConvertUIDtoString((UINT8*)(&pMsgIn->msgData.dataInd.dstAddr), buf, pMsgIn->msgData.dataInd.srcAddrMode), dstpanid, pMsgIn->msgData.dataInd.srcAddrMode);
        }
    }   else    {
        // Note: The SMAC stack broadcast pkt is not complaint with the IEEE-std.
        //          The SMAC broadcast format:
        //              AddrMode = Long
        //              Panid    = 0xFFFF
        //              Uid      = 0xFFFFFFFFFFFFFFFF
        //
        //          The IEEE std broadcast format 1:
        //              AddrMode = Short
        //              Panid    = 0xFFFF
        //              ShortAddr= 0xFFFF
        //
        //          The IEEE std broadcast format 2:
        //              AddrMode = None
        //              Panid    = 0xFFFF

#ifdef  IEEE_STD_BROADCAST
        if (pMsgIn->msgData.dataInd.dstAddrMode != gAddrModeLong_c) {
            //  check for broadcast packet: In broadcast packet the destination panid is set to 0xFFFF, addr mode is short and
            //  short address is set to 0xFFFF
            if ((dstpanid == 0xFFFF) && (dstshortaddr == 0xFFFF) && (pMsgIn->msgData.dataInd.dstAddrMode == gAddrModeShort_c))  {
#else
        memset (dstuid, 0xFF, UID_MAX_SIZE);
        if (dstpanid == 0xFFFF) {
            if (memcmp(dstuid, &pMsgIn->msgData.dataInd.dstAddr, UID_MAX_SIZE) == 0)    {
#endif
                //  Broadcast packet
                broadcast = TRUE;

#ifndef     RF_NETWORK_MASTER_DEVICE

                //  Current Booadcast restrictions:
                //      1- supports packets from designated gateway
                //      2- supports NetworkService packets
                //
                //  If restrictions are removed then the gateway restart message processing needs to be revised to accept restart
                //  broadcast only from the the designated gw's
                RFIntHostApp.pF_GetGatewayUID_PANID_ADDRMODE(&srcuidvalid, srcuid, &srcpanid, &srcaddrmode, (UINT8*)GATEWAY_DEFAULT_UID, GATEWAY_DEFAULT_PANID, GATEWAY_ADDR_MODE);
                if ( ((ConvUINTToBigIndian((UINT8*) &pMsgIn->msgData.dataInd.srcPanId) == srcpanid) &&
                      (memcmp(srcuid, &pMsgIn->msgData.dataInd.srcAddr, UID_MAX_SIZE) == 0))        ||
                      (pkthdr->type == RF_PACKET_NETWORK_SERVICES)) {
                    //  okay to process packet
                    process = TRUE;
                }

#else
                //  okay to process packet
                process = TRUE;
#endif
            }   else    {
                if (isTraceEnabled) {
                    //  Invalid or unrecognized packet, could be MNET packet
                    RFIntHostApp.pF_ConsolePrintf ("MCP packet error, invalid dst addr: uid[%s] panid[%04x] addrmode[%x]\r\n",  ConvertUIDtoString((UINT8*)(&pMsgIn->msgData.dataInd.dstAddr), buf, pMsgIn->msgData.dataInd.dstAddrMode), dstpanid, pMsgIn->msgData.dataInd.dstAddrMode);
                }
            }
        }   else    {
            //  the uid/panid was checked by the rf stack
            process = TRUE;
        }
    }

    if (process)    {
        if (SUCCESS == RFRecMsg_QualifyDataPacket (&index, pMsgIn, broadcast))  {
			//  copy packet to local buffer for processing
            result = RFRecMsg_ProcessMessagesInDataPacket (index, &pMsgIn->msgData.dataInd.msdu[sizeof(TYPE_RF_PAYLOAD_HEADER)], pMsgIn->msgData.dataInd.msduLength-sizeof(TYPE_RF_PAYLOAD_HEADER));
            if (result == SUCCESS)  {
                //      
            }   else    {
                //  log error pkt
                RFDesc_LogErrorPkt (index, pMsgIn);
            }
            //  turn on the received packet indicator
            RFIntHostApp.pF_RFReceiverIndicatorFlash();
        }
    #ifdef  RF_DBG_LOG
        RFComLog_LogEvent (RF_DBG_MCP_RX_PACKET, pMsgIn->msgData.dataCnf.msduHandle, pMsgIn->msgType, 0, pMsgIn->msgData.dataInd.msdu[sizeof(TYPE_RF_PAYLOAD_HEADER)+1], g_pRFDeviceDescriptiveTable[index].COMLog.rxpacketcount, OS_RetrieveClock());
    #endif
    }   else    {
    #ifdef  RF_DBG_LOG
        RFComLog_LogEvent (RF_DBG_MCP_RX_PACKET_NOT_FORME, pMsgIn->msgData.dataCnf.msduHandle, pMsgIn->msgType, 0, pMsgIn->msgData.dataInd.msdu[sizeof(TYPE_RF_PAYLOAD_HEADER)+1], 0, OS_RetrieveClock());
    #endif
    }

    return (result);
}


////////////////////////////////////////////////////////////////////////////////
//  GLOBAL FUNCTIONS
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function processes the MLM messages
 *
 * \detail
 * \date        09/05/2006
 * \param       UINT8 index, nwkMessage_t *pMsgIn
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 */
UINT16 RFRecMsg_MLMEMessageProcessing (UINT8 index, nwkMessage_t *pMsgIn)   {
    UINT16  result = SUCCESS;
    INT8    uidstr[UID_MAX_SIZE*3];
    BOOL    isTraceEnabled;

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

    switch(pMsgIn->msgType) {
        //  Requests from the network
        case    gNwkAssociateInd_c:
            //  Reply to association request
            RFTransmit_BuildAssociateResponseMessageAndPostToPipe((UINT8*) &pMsgIn->msgData.associateInd.deviceAddress);
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Association request was received from %s\r\n", ConvertUIDtoString ((UINT8*) &pMsgIn->msgData.associateInd.deviceAddress[0], uidstr, gAddrModeLong_c));
            }
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_MLME_RX_PACKET, index, pMsgIn->msgType, 0, 0, 0, OS_RetrieveClock());
            #endif
            break;
        case    gNwkDisassociateInd_c:
            //  Reply to disassociate request
            if (isTraceEnabled) {
                RFIntHostApp.pF_ConsolePrintf ("Disassociate request was received from %s\r\n", ConvertUIDtoString ((UINT8*) &pMsgIn->msgData.disassociateInd.deviceAddress[0], uidstr, gAddrModeLong_c));
            }
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_MLME_RX_PACKET, index, pMsgIn->msgType, 0, 0, 0, OS_RetrieveClock());
            #endif
            break;
        case    gNwkGtsInd_c:
        case    gNwkBeaconNotifyInd_c:
        case    gNwkOrphanInd_c:
        case    gNwkCommStatusInd_c:
        case    gNwkSyncLossInd_c:
        case    gNwkBeaconStartInd_c:
            //  Requests
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_MLME_RX_PACKET, index, pMsgIn->msgType, 0, 0, 0, OS_RetrieveClock());
            #endif
            break;

        //  Confirmation from the network
        case    gNwkDisassociateCnf_c:
        case    gNwkAssociateCnf_c:
        case    gNwkScanCnf_c:
        case    gNwkGtsCnf_c:
        case    gNwkGetCnf_c:
        case    gNwkResetCnf_c:
        case    gNwkRxEnableCnf_c:
        case    gNwkSetCnf_c:
        case    gNwkStartCnf_c:
        case    gNwkPollCnf_c:
        case    gNwkErrorCnf_c:
        case    gNwkMaintenanceScanCnf_c:
			result = RFResultCode_MLMEProcessing (index, pMsgIn);
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_MLME_RX_CONF_PACKET, index, pMsgIn->msgType, 0, 0, 0, OS_RetrieveClock());
            #endif
            break;

        case    gNwkMaxPrimitives_c:
            break;
    }

    return (result);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function processes the ASP message
 *
 * \detail
 * \date        09/05/2006
 *
 * \param       UINT8 index, aspToAppMsg_t * pMsgIn
 *
 * \return      UINT16 result
 *
 * \retval      SUCCESS/FAILURE
 */
UINT16 RFRecMsg_ASPMessageProcessing (UINT8 index, aspToAppMsg_t * pMsgIn)  {
    UINT16 result  = SUCCESS;

    switch(pMsgIn->msgType) {
        case  gAspAppGetTimeCfm_c:
        case  gAspAppGetInactiveTimeCfm_c:
        case  gAspAppDozeCfm_c:
        case  gAspAppAutoDozeCfm_c:
        case  gAspAppHibernateCfm_c:
        case  gAspAppWakeCfm_c:
        case  gAspAppEventCfm_c:
        case  gAspAppTrimCfm_c:
        case  gAspAppDdrCfm_c:
        case  gAspAppPortCfm_c:
        case  gAspAppClkoCfm_c:
        case  gAspAppTempCfm_c:
        case  gAspAppNvRamCfm_c:
        case  gAspAppBatteryCfm_c:
        case  gAspAppSetNotifyCfm_c:
        case  gAspAppSetMinDozeTimeCfm_c:
        case  gAspAppTelecTestCfm_c:
        case  gAspAppTelecSetChannelCfm_c:
        case  gAspAppSetPowerLevelCfm_c:
        case  gAspAppAcomaCfm_c:
        case  gAspAppGetMacStateCfm_c:
        case  gAspErrorCfm_c:
			result = RFResultCode_ASPProcessing (index, pMsgIn->msgType);
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_ASP_RX_CONF_PACKET, index, pMsgIn->msgType, 0, 0, 0, OS_RetrieveClock());
            #endif
            break;

        //  Events from ASP to Application
        case  gAspAppWakeInd_c:
        case  gAspAppIdleInd_c:
        case  gAspAppInactiveInd_c:
        case  gAspAppEventInd_c:
        case  gAspMaxPrimitives_c:
            //  Process incoming event
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_ASP_RX_PACKET, index, pMsgIn->msgType, 0, 0, 0, OS_RetrieveClock());
            #endif
            break;

        default:
            result = FAILURE;
            //  Process incoming event
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_ASP_RX_PACKET, index, pMsgIn->msgType, 0, 0, 0, OS_RetrieveClock());
            #endif
            break;
    }

    return (result);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function displays the received MCP message packets on console
 *
 * \detail
 * \date        01/017/2007
 *
 * \param       mcpsToNwkMessage_t *pMsgIn, BOOL promiscousmode
 *
 * \return      None
 *
 * \retval
 */
void    RFRecMsg_ShowMCPMessage (mcpsToNwkMessage_t *pMsgIn, BOOL promiscousmode)   {
    UINT16  i, j, l;
    INT8    buf1[600], buf2[400];

    if (promiscousmode) {
        RFIntHostApp.pF_ConsolePrintf ("MCP PROMISC RX: Len[%02x] LQ[%02x] ",  pMsgIn->msgData.promInd.msduLength, pMsgIn->msgData.promInd.mpduLinkQuality);
        RFIntHostApp.pF_ConsolePrintf ("data[");
        for (buf1[0]=0, i=0; i<pMsgIn->msgData.promInd.msduLength; i++) {
            sprintf((INT8*)&buf1[i*2], "%02x", pMsgIn->msgData.promInd.msdu[i]);
        }
        RFIntHostApp.pF_ConsolePrintf ("%s]\r\n", buf1);
    }   else    {
        RFIntHostApp.pF_ConsolePrintf ("MCP RX: dst[%s %02x%02x %02x] src[%s %02x%02x %02x] Len[%02x] LQ[%02x] Security[%x] aclEntry[%02x]",
                    ConvertUIDtoString((UINT8*) &pMsgIn->msgData.dataInd.dstAddr, buf1, pMsgIn->msgData.dataInd.dstAddrMode),
                    pMsgIn->msgData.dataInd.dstPanId[0], pMsgIn->msgData.dataInd.dstPanId[1], pMsgIn->msgData.dataInd.dstAddrMode,
                    ConvertUIDtoString((UINT8*) &pMsgIn->msgData.dataInd.srcAddr, buf2, pMsgIn->msgData.dataInd.srcAddrMode),
                    pMsgIn->msgData.dataInd.srcPanId[0], pMsgIn->msgData.dataInd.srcPanId[1], pMsgIn->msgData.dataInd.srcAddrMode,
                    pMsgIn->msgData.dataInd.msduLength, pMsgIn->msgData.dataInd.mpduLinkQuality, pMsgIn->msgData.dataInd.securityUse,
                    pMsgIn->msgData.dataInd.aclEntry);

        for (i=0, buf1[0]=0; (i<pMsgIn->msgData.dataInd.msduLength) && (i<4); ) {
            sprintf ((INT8*)&buf1[i*2], "%02x", pMsgIn->msgData.dataInd.msdu[i]);
            i++;
        }
        RFIntHostApp.pF_ConsolePrintf (" plhdr[%s]", buf1);
        RFIntHostApp.pF_ConsolePrintf (" data[");
        for (buf1[0]=0, l=pMsgIn->msgData.dataInd.msdu[i]+1, j=0; i<pMsgIn->msgData.dataInd.msduLength; i++)    {
            j += sprintf((INT8*)&buf1[j], "%02x", pMsgIn->msgData.dataInd.msdu[i]);
            if (--l == 0)   {
                l = pMsgIn->msgData.dataInd.msdu[i+1]+1;
                j += sprintf (&buf1[j], " ");
            }
        }
        RFIntHostApp.pF_ConsolePrintf ("%s]\r\n", buf1);
    }
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function processes the MCP data messages (ACKs, NAKS, Incoming data packet, Promiscouous mode data packets)
 *              The packets are then handed over to the corresponding functions
 *
 * \detail
 * \date        09/05/2006
 *
 * \param       mcpsToNwkMessage_t *pMsgIn
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 */
UINT16 RFRecMsg_MCPMessageProcessing (mcpsToNwkMessage_t *pMsgIn)   {
    BOOL isTraceEnabled;
    UINT16  result = SUCCESS;

    RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

    switch(pMsgIn->msgType) {
        case gMcpsDataCnf_c:
            //  MCP data packet transmission confirmation received
            if   (isTraceEnabled)   {
                RFIntHostApp.pF_ConsolePrintf ("MCP packet confirmation: handle[%02x] status[%02x]\r\n", pMsgIn->msgData.dataCnf.msduHandle, pMsgIn->msgData.dataCnf.status);
            }

#ifndef	RF_NETWORK_MASTER_DEVICE
            // 20120227 PM Store local device link quality for the ack so that we update the value for heartbeat messages when there is
            // no other activity.
            if (pMsgIn->msgData.dataCnf.msduHandle == DEVICE_DESCRIPTIVE_TABLE_INDEX && pMsgIn->msgData.dataCnf.status == gSuccess_c)
            {
                // Handle is the index and it has to be for the device (0), otherwise we will update this even for broadcast messages
                // we send (in smac_app.c, gSuccess_c is set for broadcast message (example: acquire)
                RFDesc_UpdateLinkQualityOfLocalDevice  (DEVICE_DESCRIPTIVE_TABLE_INDEX, pMsgIn->msgData.dataCnf.lqi);
            }
#endif
			//  Process transmitted frame result code (ACK/NAK or other errors)
            RFResultCode_MCPProcessing(pMsgIn->msgData.dataCnf.msduHandle, pMsgIn->msgData.dataCnf.status);
            break;

        case gMcpsDataInd_c:
            //  MCPS-Data packet received
            if   (isTraceEnabled)   {
                RFRecMsg_ShowMCPMessage (pMsgIn, FALSE);
            }
            //  Process packet
            result = RFRecMsg_DataPacketProcessing (pMsgIn);
            break;

        case gMcpsPurgeCnf_c:
            //  MCP purged data packet transmission confirmation received
            if   (isTraceEnabled)   {
                RFIntHostApp.pF_ConsolePrintf ("MCP purge packet confirmation: handle[%02x] status[%02x]\r\n", pMsgIn->msgData.dataCnf.msduHandle, pMsgIn->msgData.dataCnf.status);
            }
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_MCP_RX_PURGE_CONF_PACKET, pMsgIn->msgData.dataCnf.msduHandle, pMsgIn->msgType, 0, pMsgIn->msgData.dataCnf.status, 0, OS_RetrieveClock());
            #endif
            break;

        case gMcpsPromInd_c:
            //  MCP promiscuous mode packet received
            if   (isTraceEnabled)   {
                RFRecMsg_ShowMCPMessage (pMsgIn, TRUE);
            }
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_MCP_RX_PACKET, pMsgIn->msgData.dataCnf.msduHandle, pMsgIn->msgType, 0,  pMsgIn->msgData.dataInd.msdu[sizeof(TYPE_RF_PAYLOAD_HEADER)+1], 0, OS_RetrieveClock());
            #endif
            break;

        default:
            //  Unknown MCP packet type received
            RFIntHostApp.pF_ErrorPrintf ("RF Stack Error, unknown MCP packet type received msg type[%02x]", pMsgIn->msgType);
            result = FAILURE;
            #ifdef  RF_DBG_LOG
                RFComLog_LogEvent (RF_DBG_MCP_RX_PACKET_UNKNOWN, pMsgIn->msgData.dataCnf.msduHandle, pMsgIn->msgType, 0, pMsgIn->msgData.dataCnf.status, 0, OS_RetrieveClock());
            #endif
            break;
    }

    return (result);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function saves the option flags
 *
 * \detail
 * \date        05/31/2007
 * \param       UINT8 options
 *
 * \return      None
 *
 * \retval
 */
void RFRecMsg_SaveOptionFlags (UINT8 options)   {
    OptionFlags = options;
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief       This function returns the option flag settings
 *
 * \detail
 * \date        05/31/2007
 * \param       None
 *
 * \return      UINT8
 *
 * \retval      options
 */
UINT8   RFRecMsg_GetOptionFlags (void)  {
    return(OptionFlags);
}
